/*
 * Decompiled with CFR 0.152.
 */
package com.complexible.stardog.util.report.sdu;

import com.complexible.stardog.util.report.sdu.SDULogFileMetadata;
import com.complexible.stardog.util.report.sdu.SDUUsageEvent;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Iterators;
import java.io.BufferedReader;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.zip.GZIPInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SDUUsageEventIterator
implements Iterator<SDUUsageEvent>,
Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(SDUUsageEventIterator.class);
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private static final long EARLY_TERMINATION_BUFFER_MS = 300000L;
    private static final int MAX_PARSE_WARNINGS_PER_FILE = 5;
    private final List<OpenLogFile> mOpenLogFiles;
    private final Long mStartTime;
    private final Long mEndTime;
    private final Long mEndTimeThreshold;
    private final Iterator<SDUUsageEvent> mConcatenatedFileIterator;
    private boolean mExhausted = false;

    public SDUUsageEventIterator(List<Path> theLogFiles, Long theStartTime, Long theEndTime) {
        this.mStartTime = theStartTime;
        this.mEndTime = theEndTime;
        this.mEndTimeThreshold = theEndTime != null ? Long.valueOf(theEndTime + 300000L) : null;
        this.mOpenLogFiles = new ArrayList<OpenLogFile>(theLogFiles.size());
        try {
            for (Path logFile : theLogFiles) {
                BufferedReader reader = SDUUsageEventIterator.openFile(logFile);
                this.mOpenLogFiles.add(new OpenLogFile(logFile, reader));
                LOGGER.debug("Pre-opened SDU usage log file: {}", (Object)logFile);
            }
        }
        catch (IOException e) {
            this.closeAllFiles();
            throw new UncheckedIOException("Error opening SDU usage log files", e);
        }
        this.mConcatenatedFileIterator = Iterators.concat(this.createFileIterators());
    }

    private static BufferedReader openFile(Path theLogFile) throws IOException {
        boolean isCompressed = theLogFile.getFileName().toString().endsWith(".gz");
        InputStream inputStream = isCompressed ? new GZIPInputStream(Files.newInputStream(theLogFile, new OpenOption[0])) : Files.newInputStream(theLogFile, new OpenOption[0]);
        return new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
    }

    private Iterator<Iterator<SDUUsageEvent>> createFileIterators() {
        return new Iterator<Iterator<SDUUsageEvent>>(){
            private int mIndex = 0;
            private OpenLogFile mNextFile = null;

            @Override
            public boolean hasNext() {
                if (SDUUsageEventIterator.this.mExhausted) {
                    return false;
                }
                if (this.mNextFile != null) {
                    return true;
                }
                this.mNextFile = this.findNextFile();
                return this.mNextFile != null;
            }

            @Override
            public Iterator<SDUUsageEvent> next() {
                if (!this.hasNext()) {
                    throw new NoSuchElementException();
                }
                OpenLogFile file = this.mNextFile;
                this.mNextFile = null;
                return new SingleFileIterator(file);
            }

            private OpenLogFile findNextFile() {
                while (this.mIndex < SDUUsageEventIterator.this.mOpenLogFiles.size()) {
                    OpenLogFile openFile;
                    SDULogFileMetadata.FileTimeRange timeRange;
                    if ((timeRange = (openFile = SDUUsageEventIterator.this.mOpenLogFiles.get(this.mIndex++)).getTimeRange()).getLastTimestamp() != null && SDUUsageEventIterator.this.mStartTime != null && timeRange.getLastTimestamp() < SDUUsageEventIterator.this.mStartTime) {
                        LOGGER.debug("Skipping file {} - all events before start time (last={}, start={})", new Object[]{openFile.path, timeRange.getLastTimestamp(), SDUUsageEventIterator.this.mStartTime});
                        SDUUsageEventIterator.this.closeQuietly(openFile);
                        continue;
                    }
                    if (timeRange.getFirstTimestamp() != null && SDUUsageEventIterator.this.mEndTimeThreshold != null && timeRange.getFirstTimestamp() > SDUUsageEventIterator.this.mEndTimeThreshold) {
                        LOGGER.debug("Stopping at file {} - all remaining events after end time (first={}, end={})", new Object[]{openFile.path, timeRange.getFirstTimestamp(), SDUUsageEventIterator.this.mEndTime});
                        SDUUsageEventIterator.this.closeQuietly(openFile);
                        SDUUsageEventIterator.this.mExhausted = true;
                        return null;
                    }
                    LOGGER.debug("Processing SDU usage log file: {}", (Object)openFile.path);
                    return openFile;
                }
                return null;
            }
        };
    }

    @Override
    public boolean hasNext() {
        if (this.mExhausted) {
            return false;
        }
        return this.mConcatenatedFileIterator.hasNext();
    }

    @Override
    public SDUUsageEvent next() {
        if (!this.hasNext()) {
            throw new NoSuchElementException("No more SDU usage events");
        }
        return this.mConcatenatedFileIterator.next();
    }

    @Override
    public void close() throws IOException {
        this.closeAllFiles();
        this.mExhausted = true;
    }

    private void closeAllFiles() {
        for (OpenLogFile openFile : this.mOpenLogFiles) {
            this.closeQuietly(openFile);
        }
    }

    private void closeQuietly(OpenLogFile theFile) {
        try {
            theFile.close();
        }
        catch (IOException e) {
            LOGGER.warn("Error closing SDU usage log file: {}", (Object)theFile.path, (Object)e);
        }
    }

    private boolean isWithinTimeRange(long theEpochTime) {
        if (this.mStartTime != null && theEpochTime < this.mStartTime) {
            return false;
        }
        return this.mEndTime == null || theEpochTime <= this.mEndTime;
    }

    private boolean isFileFullyWithinRange(SDULogFileMetadata.FileTimeRange theTimeRange) {
        if (this.mStartTime == null && this.mEndTime == null) {
            return true;
        }
        if (theTimeRange.getFirstTimestamp() == null || theTimeRange.getLastTimestamp() == null) {
            return false;
        }
        boolean afterStart = this.mStartTime == null || theTimeRange.getFirstTimestamp() >= this.mStartTime;
        boolean beforeEnd = this.mEndTime == null || theTimeRange.getLastTimestamp() <= this.mEndTime;
        return afterStart && beforeEnd;
    }

    private static class OpenLogFile
    implements Closeable {
        final Path path;
        final BufferedReader reader;
        SDULogFileMetadata.FileTimeRange timeRange;

        OpenLogFile(Path thePath, BufferedReader theReader) {
            this.path = thePath;
            this.reader = theReader;
        }

        SDULogFileMetadata.FileTimeRange getTimeRange() {
            if (this.timeRange == null) {
                this.timeRange = SDULogFileMetadata.getFileTimeRange(this.path);
            }
            return this.timeRange;
        }

        @Override
        public void close() throws IOException {
            this.reader.close();
        }
    }

    private class SingleFileIterator
    implements Iterator<SDUUsageEvent> {
        private final OpenLogFile mFile;
        private final boolean mSkipTimestampCheck;
        private int mParseFailures = 0;
        private SDUUsageEvent mNextEvent = null;
        private boolean mFileExhausted = false;

        SingleFileIterator(OpenLogFile theFile) {
            this.mFile = theFile;
            this.mSkipTimestampCheck = SDUUsageEventIterator.this.isFileFullyWithinRange(theFile.getTimeRange());
            if (this.mSkipTimestampCheck) {
                LOGGER.debug("File {} is fully within time range - skipping per-line timestamp checks", (Object)theFile.path);
            }
        }

        @Override
        public boolean hasNext() {
            if (SDUUsageEventIterator.this.mExhausted || this.mFileExhausted) {
                return false;
            }
            if (this.mNextEvent != null) {
                return true;
            }
            this.mNextEvent = this.fetchNextEvent();
            return this.mNextEvent != null;
        }

        @Override
        public SDUUsageEvent next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            SDUUsageEvent event = this.mNextEvent;
            this.mNextEvent = null;
            return event;
        }

        private SDUUsageEvent fetchNextEvent() {
            try {
                String line;
                while ((line = this.mFile.reader.readLine()) != null) {
                    Long epochTime;
                    if (line.isBlank()) continue;
                    if (!(this.mSkipTimestampCheck || SDUUsageEventIterator.this.mStartTime == null && SDUUsageEventIterator.this.mEndTime == null || (epochTime = SDULogFileMetadata.extractEpoch(line)) == null)) {
                        if (SDUUsageEventIterator.this.mEndTimeThreshold != null && epochTime > SDUUsageEventIterator.this.mEndTimeThreshold) {
                            LOGGER.debug("Early termination in {}: event time {} past end threshold", (Object)this.mFile.path, (Object)epochTime);
                            SDUUsageEventIterator.this.mExhausted = true;
                            this.mFileExhausted = true;
                            return null;
                        }
                        if (!SDUUsageEventIterator.this.isWithinTimeRange(epochTime)) continue;
                    }
                    try {
                        return (SDUUsageEvent)objectMapper.readValue(line, SDUUsageEvent.class);
                    }
                    catch (JsonProcessingException e) {
                        ++this.mParseFailures;
                        if (this.mParseFailures > 5) continue;
                        LOGGER.warn("Failed to parse SDU usage log line in {}: {}", new Object[]{this.mFile.path, line, e});
                    }
                }
                if (this.mParseFailures > 5) {
                    LOGGER.warn("Suppressed {} additional parse failure warnings in {}", (Object)(this.mParseFailures - 5), (Object)this.mFile.path);
                }
                this.mFileExhausted = true;
                return null;
            }
            catch (IOException e) {
                throw new UncheckedIOException("Error reading SDU usage log file: " + String.valueOf(this.mFile.path), e);
            }
        }
    }
}

