/*
 * Decompiled with CFR 0.152.
 */
package com.complexible.memory.memoryblock;

import com.complexible.common.base.Disposable;
import com.complexible.common.base.Disposables;
import com.complexible.memory.memoryblock.HeapMemoryBlock;
import com.complexible.memory.memoryblock.MemoryBlock;
import com.complexible.memory.memoryblock.MemoryPoolMonitor;
import com.complexible.memory.memoryblock.MemoryType;
import com.complexible.memory.memoryblock.NativeMemoryBlock;
import com.complexible.memory.util.MMBits;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.lang3.mutable.MutableInt;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class MemoryBlockPool
implements Disposable {
    private static final Logger LOGGER = LoggerFactory.getLogger(MemoryBlockPool.class);
    private final int mBlockSize;
    private final MemoryPoolMonitor mMonitor;
    private final Queue<MemoryBlock> mSegmentQueue;
    private final AtomicLong mApproximateSegmentQueueSize = new AtomicLong(0L);
    private final ThreadLocal<Deque<MemoryBlock>> mThreadLocalPool = new ThreadLocal();
    private final ThreadLocal<MutableInt> mThreadLocalPoolCounter = new ThreadLocal();

    public MemoryBlockPool(int theBlockSize, MemoryPoolMonitor theMemoryPool, boolean theAllocateImmediately) {
        this.mBlockSize = theBlockSize;
        this.mMonitor = theMemoryPool;
        this.mSegmentQueue = new ConcurrentLinkedQueue<MemoryBlock>();
        if (theAllocateImmediately) {
            this.allocateAll();
        }
        Disposables.markCreated((Object)this);
    }

    public MemoryBlock getNextMemoryBlock() {
        Deque<MemoryBlock> aLocalBlocks = this.mThreadLocalPool.get();
        if (aLocalBlocks != null && !aLocalBlocks.isEmpty()) {
            return aLocalBlocks.poll();
        }
        return this.getNextGlobalMemoryBlock();
    }

    public int allocateLocalBlocks(int theAcquiredNumberOfBlocks) {
        MemoryBlock aMemoryBlock;
        int aAllocatedBlockNumber;
        if (this.mThreadLocalPool.get() == null) {
            this.mThreadLocalPool.set(new ArrayDeque());
            this.mThreadLocalPoolCounter.set(new MutableInt(0));
        }
        for (aAllocatedBlockNumber = 0; aAllocatedBlockNumber < theAcquiredNumberOfBlocks && (aMemoryBlock = this.getNextGlobalMemoryBlock()) != null; ++aAllocatedBlockNumber) {
            this.mThreadLocalPool.get().add(aMemoryBlock);
            this.mThreadLocalPoolCounter.get().increment();
        }
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("{} block pool: allocated {} local blocks", (Object)this.mMonitor.getMemoryType(), (Object)aAllocatedBlockNumber);
        }
        return aAllocatedBlockNumber;
    }

    public void releaseMemoryBlock(MemoryBlock theMemoryBlock) {
        if (this.tryLocalRelease(theMemoryBlock)) {
            return;
        }
        this.releaseBlockToQueue(theMemoryBlock);
    }

    private void releaseBlockToQueue(MemoryBlock theMemoryBlock) {
        this.mSegmentQueue.offer(theMemoryBlock);
        this.mApproximateSegmentQueueSize.incrementAndGet();
    }

    public void releaseLocalBlocks() {
        MutableInt aMutableCounter = this.mThreadLocalPoolCounter.get();
        if (aMutableCounter != null) {
            this.releaseLocalBlocks(aMutableCounter.intValue());
        }
    }

    public void releaseLocalBlocks(int theBlocksCount) {
        Deque<MemoryBlock> aMemoryBlockDeque = this.mThreadLocalPool.get();
        if (aMemoryBlockDeque == null) {
            return;
        }
        for (int i = 0; i < theBlocksCount; ++i) {
            MemoryBlock aMemoryBlock = aMemoryBlockDeque.poll();
            if (aMemoryBlock == null) {
                this.mThreadLocalPool.remove();
                this.mThreadLocalPoolCounter.remove();
                return;
            }
            this.mThreadLocalPoolCounter.get().decrement();
            this.releaseBlockToQueue(aMemoryBlock);
        }
        if (this.mThreadLocalPoolCounter.get().getValue() == 0) {
            this.mThreadLocalPool.remove();
            this.mThreadLocalPoolCounter.remove();
        }
    }

    public long getTotal() {
        return this.mMonitor.getTotal();
    }

    public long getUsed() {
        return this.mMonitor.getUsed();
    }

    public void dispose() {
        Disposables.markReleased((Object)this);
        long aReleased = 0L;
        while (true) {
            this.releaseLocalBlocks();
            MemoryBlock aMemoryBlock = this.mSegmentQueue.poll();
            if (aMemoryBlock == null) {
                if (aReleased > 0L) {
                    MMBits.releaseMemory(aReleased);
                }
                return;
            }
            if (aMemoryBlock.type() == MemoryType.NATIVE) {
                aReleased += (long)aMemoryBlock.getTotalBlockSize();
            }
            this.mApproximateSegmentQueueSize.decrementAndGet();
            this.mMonitor.release(aMemoryBlock.getTotalBlockSize());
            aMemoryBlock.dispose();
        }
    }

    private void allocateAll() {
        long aAllocated = 0L;
        while (this.mMonitor.reserve(MemoryBlock.computeTotalBlockSize(this.mBlockSize))) {
            MemoryBlock aMemoryBlock = this.createBlock();
            this.releaseBlockToQueue(aMemoryBlock);
            if (aMemoryBlock.type() != MemoryType.NATIVE) continue;
            aAllocated += (long)aMemoryBlock.getTotalBlockSize();
        }
        if (this.mMonitor.getMemoryType() == MemoryType.NATIVE && aAllocated > 0L) {
            MMBits.reserveMemory(aAllocated);
        }
    }

    private MemoryBlock getNextGlobalMemoryBlock() {
        MemoryBlock aMemoryBlock = this.mSegmentQueue.poll();
        if (aMemoryBlock != null) {
            this.mApproximateSegmentQueueSize.decrementAndGet();
            return aMemoryBlock;
        }
        if (!this.mMonitor.reserve(MemoryBlock.computeTotalBlockSize(this.mBlockSize))) {
            return null;
        }
        aMemoryBlock = this.createBlock();
        MMBits.reserveMemory(aMemoryBlock.getTotalBlockSize());
        return aMemoryBlock;
    }

    private MemoryBlock createBlock() {
        return this.mMonitor.getMemoryType() == MemoryType.HEAP ? new HeapMemoryBlock(this.mBlockSize) : new NativeMemoryBlock(this.mBlockSize);
    }

    private boolean tryLocalRelease(MemoryBlock theMemoryBlock) {
        MutableInt aPoolCounter;
        Deque<MemoryBlock> aMemoryBlockDeque = this.mThreadLocalPool.get();
        if (aMemoryBlockDeque != null && (aPoolCounter = this.mThreadLocalPoolCounter.get()).intValue() - aMemoryBlockDeque.size() > 0) {
            aMemoryBlockDeque.add(theMemoryBlock);
            return true;
        }
        return false;
    }

    public long getLocalBlockNumber() {
        Deque<MemoryBlock> aLocalPool = this.mThreadLocalPool.get();
        return aLocalPool == null ? 0L : (long)aLocalPool.size();
    }

    public long getAvailableMemory() {
        return this.availableBlockCount() * (long)MemoryBlock.computeTotalBlockSize(this.mBlockSize) + this.mMonitor.getFree();
    }

    private long availableBlockCount() {
        Deque<MemoryBlock> aLocalPool = this.mThreadLocalPool.get();
        if (aLocalPool == null) {
            return this.mApproximateSegmentQueueSize.get();
        }
        return this.mApproximateSegmentQueueSize.get() + (long)aLocalPool.size();
    }

    public MemoryType getMemoryType() {
        return this.mMonitor.getMemoryType();
    }
}

