/*
 * Decompiled with CFR 0.152.
 */
package com.complexible.stardog.cli.impl.tx;

import com.complexible.common.base.Memory;
import com.complexible.stardog.StardogException;
import com.complexible.stardog.api.admin.AdminConnection;
import com.complexible.stardog.cli.CliException;
import com.complexible.stardog.cli.PasswordReader;
import com.complexible.stardog.cli.admin.SecureStardogAdminCommand;
import com.complexible.stardog.cli.impl.tx.RangeFilterLogHandler;
import com.complexible.stardog.index.IndexOptions;
import com.complexible.stardog.metadata.MetaProperty;
import com.complexible.tx.api.logging.LogHandler;
import com.complexible.tx.api.logging.TxLogRecord;
import com.complexible.tx.api.logging.impl.TxLogs;
import com.complexible.tx.api.logging.impl.disk.DiskTxLogs;
import com.complexible.tx.api.logging.impl.navigable.NavigableDiskTxLog;
import com.google.common.base.Strings;
import com.google.inject.Inject;
import io.airlift.command.Arguments;
import io.airlift.command.Command;
import io.airlift.command.Option;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import org.apache.commons.lang3.mutable.MutableBoolean;

@Command(name="replay", description="Replay the transaction log contents.", discussion="Replays transactions from a transaction log file onto an existing database. By default, validation ensures log continuity by checking that a transaction in the log matches the last committed transaction in the database (i.e., they share the same parent transaction UUID). UUID-based filtering requires both --from-uuid and --to-uuid to be specified (open-ended ranges are not supported). If --from-uuid does not match the last committed transaction in the database (or an earlier one) validation will fail. ", examples={"* Replay a local transaction log to the server (validates by default):", "    $ stardog-admin tx replay mydb /path/to/txlog.log", "", "* Replay without validation (not recommended):", "    $ stardog-admin tx replay --skip-validate mydb txlog.log", "", "* Preview replay without applying (dry-run):", "    $ stardog-admin tx replay --dry-run mydb txlog.log", "", "* Replay a specific UUID range (both bounds required):", "    $ stardog-admin tx replay --from-uuid a1b2c3d4-... --to-uuid f9e8d7c6-... mydb txlog.log", "", "* Replay a specific time range, use with great care to ensure continuity:", "    $ stardog-admin tx replay --skip-validate --from-time 2024-01-15T10:30:00Z --to-time 2024-01-16T10:30:00Z mydb txlog.log"})
public class TxLogReplayCommand
extends SecureStardogAdminCommand<Void> {
    @Arguments(description="Database name and transaction log file path", required=true, title={"database, log-file"})
    public List<String> mArgs;
    @Option(name={"--skip-validate"}, description="Skip validation of log continuity (not recommended)")
    public boolean mSkipValidate = false;
    @Option(name={"--from-uuid"}, description="Start replaying from this transaction UUID")
    public String mFromUuid;
    @Option(name={"--to-uuid"}, description="Replay up to this transaction UUID (inclusive)")
    public String mToUuid;
    @Option(name={"--from-time"}, description="Start time for filtering (ISO-8601 format, e.g., 2024-01-15T10:30:00Z)")
    public String mFromTime;
    @Option(name={"--to-time"}, description="End time for filtering (ISO-8601 format, e.g., 2024-01-15T10:30:00Z)")
    public String mToTime;
    @Option(name={"--dry-run"}, description="Preview what would be replayed without actually applying changes. Note: validation is performed at dry-run time; if other transactions commit before the actual replay, validation may fail.")
    public boolean mDryRun = false;
    private static final LogHandler NO_OP_HANDLER = record -> {};

    @Inject
    public TxLogReplayCommand(PasswordReader theReader) {
        super(theReader);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public void performSecure(AdminConnection theConn) throws Exception {
        if (this.mArgs == null || this.mArgs.size() < 2) {
            throw new CliException(TxLogReplayCommand.get("tx.replay.missing.args", new Object[0]));
        }
        String database = this.mArgs.get(0);
        String logFilePath = this.mArgs.get(1);
        if (Strings.isNullOrEmpty((String)database) || !theConn.list().contains(database)) {
            throw new CliException(TxLogReplayCommand.get("tx.log.missing.db", database));
        }
        Path logPath = Paths.get(logFilePath, new String[0]);
        if (!Files.exists(logPath, new LinkOption[0])) {
            throw new CliException(TxLogReplayCommand.get("file.not.found", logFilePath));
        }
        if (!Files.isRegularFile(logPath, new LinkOption[0])) {
            throw new CliException(TxLogReplayCommand.get("tx.log.file.not.regular", logFilePath));
        }
        boolean validate = !this.mSkipValidate;
        AdminConnection.TxLogRange.Builder range = AdminConnection.TxLogRange.builder();
        range.startDate(this.parseInstant(this.mFromTime, "--from-time"));
        range.endDate(this.parseInstant(this.mToTime, "--to-time"));
        range.start(this.parseUuid(this.mFromUuid, "--from-uuid"));
        range.end(this.parseUuid(this.mToUuid, "--to-uuid"));
        Path replayLogPath = logPath;
        Path tempFilteredLog = null;
        try {
            boolean hasTimeRange;
            AdminConnection.TxLogRange builtRange = range.build();
            boolean hasUuidRange = builtRange.startTxId != null || builtRange.endTxId != null;
            boolean bl = hasTimeRange = builtRange.startDate != null || builtRange.endDate != null;
            if (hasUuidRange && (builtRange.startTxId == null || builtRange.endTxId == null)) {
                throw new CliException(TxLogReplayCommand.get("tx.replay.uuid.range.incomplete", new Object[0]));
            }
            if (hasUuidRange && hasTimeRange) {
                throw new CliException(TxLogReplayCommand.get("tx.replay.range.no.mixing", new Object[0]));
            }
            if (hasUuidRange) {
                replayLogPath = tempFilteredLog = this.filterLog(logPath, builtRange);
            }
            if (this.mDryRun) {
                this.performDryRun(theConn, replayLogPath, database);
            } else {
                String message = theConn.replayTransactionLog(database, replayLogPath, validate);
                System.out.println(TxLogReplayCommand.get("tx.replay.success", database, message));
            }
            if (tempFilteredLog == null) return;
        }
        catch (Throwable throwable) {
            if (tempFilteredLog == null) throw throwable;
            Files.deleteIfExists(tempFilteredLog);
            throw throwable;
        }
        Files.deleteIfExists(tempFilteredLog);
    }

    private Instant parseInstant(String input, String optionName) {
        if (input == null) {
            return null;
        }
        try {
            return Instant.parse(input);
        }
        catch (Exception e) {
            throw new CliException(TxLogReplayCommand.get("tx.log.invalid.instant", optionName, input));
        }
    }

    private UUID parseUuid(String uuidStr, String optionName) {
        if (Strings.isNullOrEmpty((String)uuidStr)) {
            return null;
        }
        try {
            return UUID.fromString(uuidStr);
        }
        catch (IllegalArgumentException e) {
            throw new CliException(TxLogReplayCommand.get("tx.replay.invalid.uuid", optionName, uuidStr));
        }
    }

    private void performDryRun(AdminConnection theConnection, Path logPath, String database) throws IOException {
        UUID lastTx;
        System.out.println(TxLogReplayCommand.get("tx.replay.dry.run.header", new Object[0]));
        UUID uUID = lastTx = this.mSkipValidate ? null : (UUID)theConnection.get(database, (MetaProperty)IndexOptions.LAST_COMMITTED_TX);
        if (!this.mSkipValidate && lastTx == null) {
            throw new IllegalStateException(TxLogReplayCommand.get("tx.replay.dry.run.last.committed", database));
        }
        MutableBoolean foundStarted = new MutableBoolean(false);
        MutableBoolean foundCommit = new MutableBoolean(false);
        HashSet uuids = new HashSet();
        LogHandler counter = record -> {
            if (record.getType() == TxLogRecord.RecordType.Started) {
                UUID uuid = Objects.requireNonNull(record.getUUID(), "Transaction UUID should not be null");
                uuids.add(uuid);
                if (lastTx != null && lastTx.equals(uuid)) {
                    foundStarted.setValue(true);
                }
            } else if (record.getType() == TxLogRecord.RecordType.Done) {
                UUID uuid = Objects.requireNonNull(record.getUUID(), "Transaction UUID should not be null");
                if (lastTx != null && lastTx.equals(uuid)) {
                    if (!foundStarted.isTrue()) {
                        throw new StardogException(TxLogReplayCommand.get("tx.replay.dry.run.validation.failure.missing.started", lastTx));
                    }
                    foundCommit.setValue(true);
                }
                if (!uuids.contains(uuid) && foundCommit.isTrue()) {
                    throw new StardogException(TxLogReplayCommand.get("tx.replay.dry.run.validation.failure.incomplete", lastTx, record.getUUID()));
                }
            }
        };
        DiskTxLogs.read(() -> Files.newInputStream(logPath, new OpenOption[0]), (LogHandler)(this.mSkipValidate ? NO_OP_HANDLER : counter));
        long fileSize = Files.size(logPath);
        System.out.println(TxLogReplayCommand.get("tx.replay.dry.run.filesize", Memory.readable((long)fileSize)));
        System.out.println(TxLogReplayCommand.get("tx.replay.dry.run.txcount", uuids.size()));
        if (!this.mSkipValidate) {
            if (!foundStarted.isTrue()) {
                throw new StardogException(TxLogReplayCommand.get("tx.replay.dry.run.validation.failure.missing.started", lastTx));
            }
            if (!foundCommit.isTrue()) {
                throw new StardogException(TxLogReplayCommand.get("tx.replay.dry.run.validation.failure.missing.done", lastTx));
            }
        }
        System.out.println(TxLogReplayCommand.get("tx.replay.dry.run.validation.success", lastTx));
    }

    private Path filterLog(Path localLogPath, AdminConnection.TxLogRange range) throws IOException {
        int filtered;
        Path filteredLog = Files.createTempFile("stardog-tx-log-filtered-", ".log", new FileAttribute[0]);
        try (NavigableDiskTxLog diskTxLog = TxLogs.buildDiskLog().bufferSize(10240).build(filteredLog);){
            RangeFilterLogHandler handler = new RangeFilterLogHandler(range, e -> {
                try {
                    diskTxLog.append(e);
                }
                catch (IOException ioException) {
                    throw new StardogException("Error writing filtered tx log", (Throwable)ioException);
                }
            });
            DiskTxLogs.read(() -> Files.newInputStream(localLogPath, new OpenOption[0]), (LogHandler)handler);
            filtered = handler.getFilteredCount();
        }
        if (filtered == 0) {
            Files.deleteIfExists(filteredLog);
            throw new CliException(TxLogReplayCommand.get("tx.replay.empty.range", new Object[0]));
        }
        return filteredLog;
    }
}

