/*
 * Decompiled with CFR 0.152.
 */
package com.complexible.stardog.cli.impl.tx;

import com.complexible.common.base.Change;
import com.complexible.common.rdf.StatementSource;
import com.complexible.stardog.StardogException;
import com.complexible.stardog.api.admin.AdminConnection;
import com.complexible.stardog.cli.CliException;
import com.complexible.stardog.cli.PasswordReader;
import com.complexible.stardog.cli.admin.SecureStardogAdminCommand;
import com.complexible.stardog.cli.impl.tx.RangeFilterLogHandler;
import com.complexible.stardog.db.tx.IndexChange;
import com.complexible.tx.api.logging.ClosableLogHandler;
import com.complexible.tx.api.logging.LogHandler;
import com.complexible.tx.api.logging.TxLogRecord;
import com.complexible.tx.api.logging.impl.PrintingLogHandler;
import com.complexible.tx.api.logging.impl.disk.DiskTxLogs;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.primitives.Ints;
import com.google.inject.Inject;
import io.airlift.command.Arguments;
import io.airlift.command.Command;
import io.airlift.command.Option;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

@Command(name="log", description="List the transaction log contents.", discussion="Export the transaction log contents from the server or read an existing transaction log from a file. Supports UUID-based filtering and multiple output formats (text for human-readable, raw for binary export).", examples={"* Read a transaction log from the server:", "    $ stardog-admin tx log mydb --format text", "", "* Export raw transaction log to a file:", "    $ stardog-admin tx log mydb --format raw --output txlog-backup.log", "", "* Read a transaction log from a local file in raw format:", "    $ stardog-admin tx log --file /path/to/raw/tx/txlog-backup.log --format text", "", "* Filter by UUID range:", "    $ stardog-admin tx log mydb --from-uuid a1b2c3d4-... --to-uuid f9e8d7c6-... --format text"})
public class TxLogExportCommand
extends SecureStardogAdminCommand<Void> {
    @Arguments(description="Database name or transaction log file path", title={"source"})
    public String mSource;
    @Option(name={"--file"}, description="Force treating source as file path")
    public boolean mForceFile = false;
    @Option(name={"--from-uuid"}, description="Start transaction UUID for filtering")
    public String mFromUuid;
    @Option(name={"--to-uuid"}, description="End transaction UUID for filtering")
    public String mToUuid;
    @Option(name={"--from-time"}, description="Start time for filtering (ISO-8601 format, e.g., 2024-01-15T10:30:00Z)")
    public String mFromTime;
    @Option(name={"--to-time"}, description="End time for filtering (ISO-8601 format, e.g., 2024-01-15T10:30:00Z)")
    public String mToTime;
    @Option(name={"--format"}, description="Output format: 'text' for human-readable or 'raw' for binary export (default: text)")
    public String mFormat = "text";
    @Option(name={"--updates"}, description="Show individual update entries from the log in the text output.")
    public boolean mShowUpdates = false;
    @Option(name={"--output", "-o"}, description="Output file path (default: stdout for text, required for raw)")
    public String mOutputFile;

    @Inject
    public TxLogExportCommand(PasswordReader theReader) {
        super(theReader);
    }

    @Override
    public void performSecure(AdminConnection theConn) throws Exception {
        if (Strings.isNullOrEmpty((String)this.mSource)) {
            throw new CliException(TxLogExportCommand.get("tx.log.missing.source", new Object[0]));
        }
        AdminConnection.TxLogRange.Builder range = AdminConnection.TxLogRange.builder();
        range.startDate(this.parseInstant(this.mFromTime, "--from-time"));
        range.endDate(this.parseInstant(this.mToTime, "--to-time"));
        range.start(this.parseUuid(this.mFromUuid, "--from-uuid"));
        range.end(this.parseUuid(this.mToUuid, "--to-uuid"));
        if (!this.mFormat.equals("text") && !this.mFormat.equals("raw")) {
            throw new CliException(TxLogExportCommand.get("tx.log.format.invalid", this.mFormat, "'text' or 'raw'"));
        }
        if (this.mFormat.equals("raw") && this.mOutputFile == null) {
            throw new CliException(TxLogExportCommand.get("tx.log.raw.output.required", new Object[0]));
        }
        boolean isFile = this.isSourceFile();
        if (isFile) {
            this.handleLocalFile(range.build());
        } else {
            this.handleRemoteServer(theConn, range.build());
        }
    }

    private UUID parseUuid(String uuidStr, String optionName) {
        if (uuidStr == null) {
            return null;
        }
        try {
            return UUID.fromString(uuidStr);
        }
        catch (IllegalArgumentException e) {
            throw new CliException(TxLogExportCommand.get("tx.log.invalid.uuid", optionName, uuidStr));
        }
    }

    private Instant parseInstant(String input, String optionName) {
        if (input == null) {
            return null;
        }
        try {
            return Instant.parse(input);
        }
        catch (Exception e) {
            throw new CliException(TxLogExportCommand.get("tx.log.invalid.instant", optionName, input));
        }
    }

    private boolean isSourceFile() {
        if (this.mForceFile) {
            return true;
        }
        if (this.mSource.startsWith("file://")) {
            return true;
        }
        Path path = Paths.get(this.mSource, new String[0]);
        return Files.exists(path, new LinkOption[0]) && Files.isRegularFile(path, new LinkOption[0]);
    }

    private void handleLocalFile(AdminConnection.TxLogRange range) throws IOException {
        Path logPath = Paths.get(this.mSource, new String[0]);
        if (!Files.exists(logPath, new LinkOption[0])) {
            throw new CliException(TxLogExportCommand.get("file.not.found", this.mSource));
        }
        if (!Files.isRegularFile(logPath, new LinkOption[0])) {
            throw new CliException(TxLogExportCommand.get("tx.log.file.not.regular", this.mSource));
        }
        if (this.mFormat.equals("raw")) {
            throw new CliException(TxLogExportCommand.get("tx.log.raw.export.not.implemented", new Object[0]));
        }
        try (ClosableLogHandler handler = this.filteringLogHandler(range);){
            DiskTxLogs.read(() -> Files.newInputStream(logPath, new OpenOption[0]), (LogHandler)handler);
        }
        catch (IOException e) {
            throw new CliException(TxLogExportCommand.get("tx.log.io.error.file", e.getMessage()), e);
        }
    }

    private ClosableLogHandler filteringLogHandler(AdminConnection.TxLogRange range) {
        FormattingLogHandler baseHandler = this.formattingLogHandler(range);
        if (range.startTxId != null && range.endTxId != null) {
            return new RangeFilterLogHandler(range, (LogHandler)baseHandler);
        }
        return baseHandler;
    }

    private FormattingLogHandler formattingLogHandler(AdminConnection.TxLogRange range) {
        if (this.mOutputFile != null) {
            Preconditions.checkNotNull((Object)this.mOutputFile, (Object)TxLogExportCommand.get("tx.log.raw.output.required", new Object[0]));
            return new FormattingLogHandler(range, Paths.get(this.mOutputFile, new String[0]));
        }
        return new FormattingLogHandler(range, new BufferedOutputStream(System.out));
    }

    private void handleRemoteServer(AdminConnection theConn, AdminConnection.TxLogRange range) throws IOException, StardogException {
        String database = this.mSource;
        if (Strings.isNullOrEmpty((String)database) || !theConn.list().contains(database)) {
            throw new CliException(TxLogExportCommand.get("tx.log.missing.db", database));
        }
        if (this.mFormat.equals("text")) {
            try (FormattingLogHandler handler = this.formattingLogHandler(range);){
                theConn.transactionLog(database, range, inputStream -> DiskTxLogs.read(() -> inputStream, (LogHandler)handler));
            }
        }
        Preconditions.checkNotNull((Object)this.mOutputFile, (Object)TxLogExportCommand.get("tx.log.raw.output.required", new Object[0]));
        Path outputPath = Paths.get(this.mOutputFile, new String[0]);
        try (OutputStream out = Files.newOutputStream(outputPath, new OpenOption[0]);){
            theConn.transactionLog(database, range, inputStream -> inputStream.transferTo(out));
        }
        System.out.println(TxLogExportCommand.get("tx.log.export.done", this.mOutputFile));
    }

    private class FormattingLogHandler
    extends PrintingLogHandler {
        private final Map<Long, Integer> mAddCounts;
        private final Map<Long, Integer> mRemoveCounts;

        public FormattingLogHandler(AdminConnection.TxLogRange range, OutputStream theOut) {
            super(theOut);
            this.mAddCounts = new HashMap<Long, Integer>();
            this.mRemoveCounts = new HashMap<Long, Integer>();
            this.printHeader(range);
        }

        public FormattingLogHandler(AdminConnection.TxLogRange range, Path theOutput) {
            super(theOutput);
            this.mAddCounts = new HashMap<Long, Integer>();
            this.mRemoveCounts = new HashMap<Long, Integer>();
            this.printHeader(range);
        }

        private void printHeader(AdminConnection.TxLogRange range) {
            this.mPrintWriter.println("Transaction Log");
            this.mPrintWriter.println("----------------------");
            this.mPrintWriter.println("Source: " + (String)(TxLogExportCommand.this.isSourceFile() ? TxLogExportCommand.this.mSource : TxLogExportCommand.this.getServerURL(TxLogExportCommand.this.mServerURL) + "/" + TxLogExportCommand.this.mSource));
            if (range.startTxId != null || range.endTxId != null) {
                if (range.startTxId != null) {
                    this.mPrintWriter.printf("From UUID: %s", range.startTxId);
                    this.mPrintWriter.print("  ");
                }
                if (range.endTxId != null) {
                    this.mPrintWriter.printf("To UUID: %s%n", range.endTxId);
                }
                this.mPrintWriter.println();
            } else {
                this.mPrintWriter.println("Range: Full");
            }
            this.mPrintWriter.println("----------------------");
            this.mPrintWriter.println();
        }

        public void handleRecord(TxLogRecord theRecord) {
            Change change;
            if (TxLogExportCommand.this.mShowUpdates || theRecord.getType() != TxLogRecord.RecordType.Update) {
                super.handleRecord(theRecord);
                return;
            }
            if (theRecord.getData() != null && theRecord.getData().getData() instanceof Change && (change = (Change)theRecord.getData().getData()).getChange() instanceof StatementSource) {
                StatementSource source = (StatementSource)change.getChange();
                assert (source.inMemory());
                int tripleCount = Ints.saturatedCast((long)source.estimatedCount());
                if (change.getChangeType() == IndexChange.Add) {
                    this.mAddCounts.merge(theRecord.getTxId(), tripleCount, Integer::sum);
                } else {
                    this.mRemoveCounts.merge(theRecord.getTxId(), tripleCount, Integer::sum);
                }
            }
        }

        protected void printRecord(TxLogRecord theRecord) {
            super.printRecord(theRecord);
            if (!TxLogExportCommand.this.mShowUpdates && theRecord.getType() == TxLogRecord.RecordType.Commit) {
                this.mPrintWriter.printf(" Added: %,d", this.mAddCounts.getOrDefault(theRecord.getTxId(), 0));
                this.mPrintWriter.printf(" Removed: %,d", this.mRemoveCounts.getOrDefault(theRecord.getTxId(), 0));
            }
        }
    }
}

